#!/bin/bash
# This script restores the original DNS settings when OpenVPN disconnects on macOS.
# It reads the backup stored in the scutil database and restores the previous configuration.

set -e
export PATH="/bin:/sbin:/usr/sbin:/usr/bin"

SCRIPT_NAME="$(basename "${0}")"
readonly SCRIPT_NAME

log_message() {
    echo "$(date '+%H:%M:%S') *openvpn_down.sh: $*" >&2
}

log_message "**********************************************"
log_message "Start of output from ${SCRIPT_NAME}"

log_message "Args: $*"
#log_message "--------------------------------------------------"
#log_message "Start ENV"
#log_message "--------------------------------------------------"
#env
#log_message "--------------------------------------------------"
#log_message "End ENV"
#log_message "--------------------------------------------------"

# Get saved info from scutil database
get_saved_info() {
    # Check if our state exists
    if ! scutil -w State:/Network/Truestack &>/dev/null -t 1 ; then
        log_message "WARNING: Not restoring network settings because no saved TrueStack DNS information was found."
        return 1
    fi

    # Get info saved by the up script
    local saved_info
    saved_info="$(scutil <<-EOF
		open
		show State:/Network/Truestack
		quit
EOF
    )"

    # Extract key values from the saved info
    SAVED_PSID="$(echo "$saved_info" | grep '^[[:space:]]*Service :' | sed -e 's/.*Service : //')"
    MADE_DNS_CHANGES="$(echo "$saved_info" | grep '^[[:space:]]*madeDnsChanges :' | sed -e 's/.*madeDnsChanges : //')"
    #SAVED_SCRIPT_NAME="$(echo "$saved_info" | grep '^[[:space:]]*ScriptName :' | sed -e 's/.*ScriptName : //')"

    if [ -z "$SAVED_PSID" ]; then
        log_message "ERROR: Could not extract saved service ID from scutil database"
        return 1
    fi

    log_message "Found saved configuration for service ID: $SAVED_PSID"
    log_message "DNS changes were made: $MADE_DNS_CHANGES"

    return 0
}

# Restore DNS configuration from scutil database
restore_dns_settings() {
    if [ "$MADE_DNS_CHANGES" = "false" ]; then
        log_message "No DNS changes were made during connection, skipping restoration"
        return 0
    fi

    log_message "Restoring original DNS configuration"

    local no_such_key="<dictionary> {
  TruestackNoSuchKey : true
}"

    # Restore State DNS configuration
    local state_dns_old
    state_dns_old="$(scutil <<-EOF
		open
		show State:/Network/Truestack/OldStateDNS
		quit
EOF
    )"

    if [ "$state_dns_old" = "$no_such_key" ]; then
        log_message "Removing State DNS configuration (none existed before)"
        scutil <<-EOF > /dev/null 2>&1 || true
			open
			remove State:/Network/Service/${SAVED_PSID}/DNS
			quit
EOF
    else
        log_message "Restoring State DNS configuration"
        scutil <<-EOF > /dev/null
			open
			get State:/Network/Truestack/OldStateDNS
			set State:/Network/Service/${SAVED_PSID}/DNS
			quit
EOF
    fi

    # Restore Setup DNS configuration
    local setup_dns_old
    setup_dns_old="$(scutil <<-EOF
		open
		show State:/Network/Truestack/OldSetupDNS
		quit
EOF
    )"

    if [ "$setup_dns_old" = "$no_such_key" ]; then
        log_message "Removing Setup DNS configuration (none existed before)"
        scutil <<-EOF > /dev/null 2>&1 || true
			open
			remove Setup:/Network/Service/${SAVED_PSID}/DNS
			quit
EOF
    else
        log_message "Restoring Setup DNS configuration"
        scutil <<-EOF > /dev/null
			open
			get State:/Network/Truestack/OldSetupDNS
			set Setup:/Network/Service/${SAVED_PSID}/DNS
			quit
EOF
    fi

    log_message "DNS configuration restored successfully"
}

# Clean up scutil database entries
cleanup_scutil_entries() {
    log_message "Cleaning up scutil database entries"

    # Remove all our stored state
    scutil <<-EOF > /dev/null 2>&1 || true
		open
		remove State:/Network/Truestack/OldStateDNS
		remove State:/Network/Truestack/OldSetupDNS
		remove State:/Network/Truestack/OldGlobalDNS
		remove State:/Network/Truestack
		quit
EOF

    log_message "Cleanup completed"
}

# Flush DNS cache
flush_dns_cache() {
    log_message "Flushing DNS cache"

    # Modern macOS
    if command -v dscacheutil >/dev/null 2>&1; then
        dscacheutil -flushcache 2>/dev/null || true
    fi

    # macOS 10.10+
    if command -v discoveryutil >/dev/null 2>&1; then
        discoveryutil udnsflushcaches 2>/dev/null || true
        discoveryutil mdnsflushcache 2>/dev/null || true
    fi

    # Notify mDNSResponder
    if command -v killall >/dev/null 2>&1; then
        killall -HUP mDNSResponder 2>/dev/null || true
        killall -HUP mDNSResponderHelper 2>/dev/null || true
    fi

    log_message "DNS cache flushed"
}

# Main execution
main() {
    if get_saved_info; then
        restore_dns_settings
        cleanup_scutil_entries
    else
        log_message "No restoration needed - no saved information found in scutil database"
    fi

    flush_dns_cache

    log_message "DNS restoration completed"
    log_message "End of output from ${SCRIPT_NAME}"
    log_message "**********************************************"

    # Log the restored configuration for debugging
    if [ "${DEBUG:-0}" = "1" ]; then
        log_message "Restored DNS configuration:"
        scutil --dns | head -20
    fi
}

# Run main function
main "$@"
