#!/bin/bash
# This script handles updating the system DNS settings for OpenVPN on macOS.
# It will save currently configured DNS servers and then apply the new ones provided by OpenVPN.
# Network data is stored in the scutil database at State:/Network/Truestack for the down script.

set -e
export PATH="/bin:/sbin:/usr/sbin:/usr/bin"

SCRIPT_NAME="$(basename "${0}")"
readonly SCRIPT_NAME

log_message() {
    echo "$(date '+%H:%M:%S') *openvpn_up.sh: $*" >&2
}

log_message "**********************************************"
log_message "Start of output from ${SCRIPT_NAME}"

log_message "Args: $*"
#log_message "--------------------------------------------------"
#log_message "Start ENV"
#log_message "--------------------------------------------------"
#env
#log_message "--------------------------------------------------"
#log_message "End ENV"
#log_message "--------------------------------------------------"

# Get the primary network service ID
get_primary_service_id() {
    scutil <<-EOF | grep 'PrimaryService : ' | sed -e 's/.*PrimaryService : //'
		open
		show State:/Network/Global/IPv4
		quit
EOF
}

PSID="$(get_primary_service_id)"
if [ -z "$PSID" ]; then
    log_message "ERROR: Could not determine primary service ID"
    exit 1
fi

log_message "Primary service ID: $PSID"

# Save current DNS configuration to scutil database
save_current_dns() {
    log_message "Backing up current DNS configuration to scutil database"

    scutil <<-EOF > /dev/null
		open

		# Back up the device's current DNS configurations
		# Indicate 'no such key' by a dictionary with a single entry: "TruestackNoSuchKey : true"
		# If there isn't a key, "TruestackNoSuchKey : true" won't be removed.
		# If there is a key, "TruestackNoSuchKey : true" will be removed and the key's contents will be used

		d.init
		d.add TruestackNoSuchKey true
		get State:/Network/Service/${PSID}/DNS
		set State:/Network/Truestack/OldStateDNS

		d.init
		d.add TruestackNoSuchKey true
		get Setup:/Network/Service/${PSID}/DNS
		set State:/Network/Truestack/OldSetupDNS

		# Store current Global DNS state for reference
		d.init
		d.add TruestackNoSuchKey true
		get State:/Network/Global/DNS
		set State:/Network/Truestack/OldGlobalDNS

		# Set the main state key with our control variables
		d.init
		d.add Service ${PSID}
		d.add madeDnsChanges "true"
		d.add ScriptName "${SCRIPT_NAME}"
		d.add SavedTimestamp "$(date)"
		set State:/Network/Truestack

		quit
EOF

    log_message "DNS configuration backed up to State:/Network/Truestack"
}

# Parse OpenVPN foreign options and extract DNS settings
parse_dns_options() {
    unset DNS_SERVERS
    unset DOMAIN_NAME
    unset SEARCH_DOMAINS

    declare -a DNS_SERVERS
    declare -a SEARCH_DOMAINS

    local option_index=1
    local dns_index=0
    local search_index=0

    while true; do
        local option_var="foreign_option_${option_index}"
        local option_value="${!option_var}"

        if [ -z "$option_value" ]; then
            break
        fi

        log_message "Processing foreign option $option_index: $option_value"

        case "$option_value" in
            "dhcp-option DOMAIN "*)
                DOMAIN_NAME="${option_value#dhcp-option DOMAIN }"
                log_message "Found domain: $DOMAIN_NAME"
                ;;
            "dhcp-option DOMAIN-SEARCH "*)
                SEARCH_DOMAINS[$search_index]="${option_value#dhcp-option DOMAIN-SEARCH }"
                log_message "Found search domain: ${SEARCH_DOMAINS[$search_index]}"
                ((search_index++))
                ;;
            "dhcp-option DNS "*)
                DNS_SERVERS[$dns_index]="${option_value#dhcp-option DNS }"
                log_message "Found DNS server: ${DNS_SERVERS[$dns_index]}"
                ((dns_index++))
                ;;
            "dhcp-option DNS6 "*)
                DNS_SERVERS[$dns_index]="${option_value#dhcp-option DNS6 }"
                log_message "Found IPv6 DNS server: ${DNS_SERVERS[$dns_index]}"
                ((dns_index++))
                ;;
        esac

        ((option_index++))
    done

    # Export arrays for use in apply_dns_settings
    export DNS_SERVERS_COUNT=${#DNS_SERVERS[@]}
    export SEARCH_DOMAINS_COUNT=${#SEARCH_DOMAINS[@]}

    for ((i=0; i<DNS_SERVERS_COUNT; i++)); do
        export "DNS_SERVER_$i=${DNS_SERVERS[$i]}"
    done

    for ((i=0; i<SEARCH_DOMAINS_COUNT; i++)); do
        export "SEARCH_DOMAIN_$i=${SEARCH_DOMAINS[$i]}"
    done

    # Store DNS settings in scutil for the down script
    local applied_dns_servers=""
    local applied_search_domains=""

    for ((i=0; i<DNS_SERVERS_COUNT; i++)); do
        local dns_var="DNS_SERVER_$i"
        if [ $i -eq 0 ]; then
            applied_dns_servers="${!dns_var}"
        else
            applied_dns_servers="${applied_dns_servers} ${!dns_var}"
        fi
    done

    for ((i=0; i<SEARCH_DOMAINS_COUNT; i++)); do
        local search_var="SEARCH_DOMAIN_$i"
        if [ $i -eq 0 ]; then
            applied_search_domains="${!search_var}"
        else
            applied_search_domains="${applied_search_domains} ${!search_var}"
        fi
    done

    # Update the main state with applied DNS settings
    scutil <<-EOF > /dev/null
		open
		get State:/Network/Truestack
		d.add AppliedDnsServers "${applied_dns_servers}"
		d.add AppliedDomainName "${DOMAIN_NAME}"
		d.add AppliedSearchDomains "${applied_search_domains}"
		set State:/Network/Truestack
		quit
EOF
}

# Apply new DNS settings
apply_dns_settings() {
    if [ "$DNS_SERVERS_COUNT" -eq 0 ]; then
        log_message "No DNS servers provided, skipping DNS configuration"

        # Update scutil to indicate no changes were made
        scutil <<-EOF > /dev/null
			open
			get State:/Network/Truestack
			d.add madeDnsChanges "false"
			set State:/Network/Truestack
			quit
EOF
        return 0
    fi

    log_message "Applying new DNS configuration"

    # Build scutil commands for DNS configuration
    local scutil_commands="open\nd.init\n"

    # Add DNS servers
    for ((i=0; i<DNS_SERVERS_COUNT; i++)); do
        local dns_var="DNS_SERVER_$i"
        scutil_commands="${scutil_commands}d.add ServerAddresses * ${!dns_var}\n"
    done

    # Add domain name
    if [ -n "$DOMAIN_NAME" ]; then
        scutil_commands="${scutil_commands}d.add DomainName ${DOMAIN_NAME}\n"
    fi

    # Add search domains
    if [ "$SEARCH_DOMAINS_COUNT" -gt 0 ]; then
        for ((i=0; i<SEARCH_DOMAINS_COUNT; i++)); do
            local search_var="SEARCH_DOMAIN_$i"
            scutil_commands="${scutil_commands}d.add SearchDomains * ${!search_var}\n"
        done
    elif [ -n "$DOMAIN_NAME" ]; then
        # If no search domains specified, use domain name as search domain
        scutil_commands="${scutil_commands}d.add SearchDomains * ${DOMAIN_NAME}\n"
    fi

    # Apply to both State and Setup (required for macOS 10.7+)
    scutil_commands="${scutil_commands}set State:/Network/Service/${PSID}/DNS\n"
    scutil_commands="${scutil_commands}set Setup:/Network/Service/${PSID}/DNS\n"
    scutil_commands="${scutil_commands}quit"

    # Execute the scutil commands
    printf "%b" "$scutil_commands" | scutil

    log_message "DNS configuration applied successfully"

    # Log the new configuration for debugging
    if [ "${DEBUG:-0}" = "1" ]; then
        log_message "New DNS configuration:"
        scutil --dns | head -20
    fi
}

# Flush DNS cache
flush_dns_cache() {
    log_message "Flushing DNS cache"

    # Modern macOS
    if command -v dscacheutil >/dev/null 2>&1; then
        dscacheutil -flushcache 2>/dev/null || true
    fi

    # macOS 10.10+
    if command -v discoveryutil >/dev/null 2>&1; then
        discoveryutil udnsflushcaches 2>/dev/null || true
        discoveryutil mdnsflushcache 2>/dev/null || true
    fi

    # Notify mDNSResponder
    if command -v killall >/dev/null 2>&1; then
        killall -HUP mDNSResponder 2>/dev/null || true
        killall -HUP mDNSResponderHelper 2>/dev/null || true
    fi

    log_message "DNS cache flushed"
}

# Main execution
main() {
    save_current_dns
    parse_dns_options
    apply_dns_settings
    flush_dns_cache

    log_message "DNS configuration completed successfully"
    log_message "End of output from ${SCRIPT_NAME}"
    log_message "**********************************************"
}

# Run main function
main "$@"
